bl_info = {
    "name": "VU.CITY Plugin v2",
    "blender": (4, 3, 2), 
    "author": "Dawud, Patrycja <support@vu.city>",
    "version": (2, 0, 0),
    "category": "Object",
    "location": "Operator Search",
    "description": "VU.CITY Plugin to facilitate model optimisation in Blender.",
}

import pathlib
import bpy


def get_all_meshes_in_hierarchy(objects):
    """Recursively gather all mesh objects, including those nested in hierarchies."""
    meshes = []

    def traverse(obj):
        if obj.type == 'MESH':
            meshes.append(obj)
        for child in obj.children:
            traverse(child)

    for obj in objects:
        traverse(obj)

    return meshes


def is_already_merged():
    """Check if only one mesh exists in the scene, indicating the function was already run."""
    meshes = [obj for obj in bpy.context.scene.objects if obj.type == 'MESH']
    return len(meshes) == 1


def ensure_single_user(meshes):
    """Ensure all mesh objects have unique data to avoid multi-user errors."""
    for mesh_obj in meshes:
        if mesh_obj.data.users > 1:
            mesh_obj.data = mesh_obj.data.copy()


class SelectMesh(bpy.types.Operator):
    """Selects the objects and merges them into one"""
    bl_idname = "object.select_mesh"
    bl_label = "Select & Merge Meshes"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        try:
            # Check if the operation was already run
            if is_already_merged():
                self.report({'WARNING'}, "The meshes are already merged. Operation skipped.")
                return {'CANCELLED'}
            # Ensure the scene contains objects
            if not bpy.context.scene.objects:
                self.report({'ERROR'}, "No objects found in the scene. Please import a model first.")
                return {'CANCELLED'}

            # Get all mesh objects, including nested ones
            meshes = get_all_meshes_in_hierarchy(bpy.context.scene.objects)

            if not meshes:
                self.report({'ERROR'}, "No meshes found in the scene. Ensure the imported model contains meshes.")
                return {'CANCELLED'}

            # Ensure all meshes are single-user to avoid multi-user errors
            ensure_single_user(meshes)

            # Switch to Object Mode
            if bpy.context.object and bpy.context.object.mode != 'OBJECT':
                bpy.ops.object.mode_set(mode='OBJECT')

            # Batch transformations and selection
            for mesh_obj in meshes:
                mesh_obj.select_set(True)
                bpy.context.view_layer.objects.active = mesh_obj

            # Apply transformations to all selected objects
            bpy.ops.object.transform_apply(location=False, rotation=False, scale=True)

            # Join all selected meshes into one
            bpy.ops.object.join()

            self.report({'INFO'}, f"Successfully processed {len(meshes)} meshes, including nested ones.")
            return {'FINISHED'}

        except Exception as e:
            # Detailed error report for unexpected issues
            self.report({'ERROR'}, f"An unexpected error occurred: {str(e)}")
            return {'CANCELLED'}


# Registering the plugin
classes = [SelectMesh]

def register():
    for cls in classes:
        bpy.utils.register_class(cls)

def unregister():
    for cls in classes:
        bpy.utils.unregister_class(cls)

if __name__ == "__main__":
    register()




class FixSmudges(bpy.types.Operator):
    """Fix smooth shading issues / dark smudges"""
    bl_idname = "object.fixing_smudges"
    bl_label = "Fix Shading Smudges"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        try:
            obj = context.object
            if obj.type == 'MESH':
                bpy.ops.object.mode_set(mode='EDIT')
                bpy.ops.mesh.select_all(action='SELECT')
                bpy.ops.mesh.customdata_custom_splitnormals_clear()  # Clear custom normals
                bpy.ops.mesh.delete_loose()  # Optional: Remove loose geometry
            try:
                if bpy.context.object and bpy.context.object.type == 'MESH':
                    bpy.ops.object.mode_set(mode='OBJECT')
                    bpy.ops.object.shade_flat()
            except RuntimeError as e:
                self.report({'ERROR'}, f"Shading failed: {e}")
                return {'CANCELLED'}
        except Exception as e:
            self.report({'ERROR'}, f"Error: {e}")
        return {'FINISHED'}

class RemoveMaterials(bpy.types.Operator):
    """Remove all materials"""
    bl_idname = "object.removing_materials"
    bl_label = "Remove Materials"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        for material in bpy.data.materials:
            material.user_clear()
            bpy.data.materials.remove(material)
        return {'FINISHED'}

class CorrectOrientation(bpy.types.Operator):
    """Flips X-rotation by 90°"""
    bl_idname = "object.correct_orientation"
    bl_label = "Rotate by 90°"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        try:
            bpy.ops.object.mode_set(mode='OBJECT')
            bpy.ops.object.select_all(action='SELECT')
            bpy.ops.object.transform_apply(location=False, rotation=True, scale=False)
            bpy.context.object.rotation_euler[0] = 1.5708
            bpy.ops.object.transform_apply(location=False, rotation=True, scale=False)
        except Exception as e:
            self.report({'ERROR'}, f"Error: {e}")
        return {'FINISHED'}

class MergeByDistance(bpy.types.Operator):
    """Collapse vertices by distance (Medium recommended)"""
    bl_idname = "object.merging_distance"
    bl_label = "Distance to merge by"
    bl_options = {'REGISTER', 'UNDO'}

    count_cm: bpy.props.FloatProperty(
        name="distance",
        description="Distance in cm to merge by"
    )

    def execute(self, context):
        try:
            bpy.ops.object.mode_set(mode='EDIT')
            bpy.ops.mesh.remove_doubles(threshold=self.count_cm)
        except Exception as e:
            self.report({'ERROR'}, f"Error: {e}")
        return {'FINISHED'}

class OriginToGeometry(bpy.types.Operator):
    """Set the object's origin to geometry"""
    bl_idname = "object.origining_geometry"
    bl_label = "Origin to Geometry"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        try:
            bpy.ops.object.origin_set(type='ORIGIN_GEOMETRY', center='BOUNDS')
        except Exception as e:
            self.report({'ERROR'}, f"Error: {e}")
        return {'FINISHED'}

class RescaleModel(bpy.types.Operator):
    """Rescale your model x0.1"""
    bl_idname = "object.rescaling_model"
    bl_label = "Scale x0.1"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        try:
            for i in range(0, 3):
                bpy.context.object.scale[i] = bpy.context.object.scale[i] * 0.1
        except Exception as e:
            self.report({'ERROR'}, f"Error: {e}")
        return {'FINISHED'}

class RescaleModel_x10(bpy.types.Operator):
    """Rescale your model x10"""
    bl_idname = "object.rescaling_model_x10"
    bl_label = "Scale x10"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        try:
            for i in range(0, 3):
                bpy.context.object.scale[i] = bpy.context.object.scale[i] * 10
        except Exception as e:
            self.report({'ERROR'}, f"Error: {e}")
        return {'FINISHED'}

class RevealFaces(bpy.types.Operator):
    """Enable face orientation to see flipped faces"""
    bl_idname = "object.revealing_faces"
    bl_label = "Reveal Flipped Faces"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        for area in bpy.context.screen.areas:
            if area.type == 'VIEW_3D':
                for space in area.spaces:
                    if space.type == 'VIEW_3D':
                        space.overlay.show_face_orientation = not space.overlay.show_face_orientation
                        break
        return {'FINISHED'}

class Triangulate(bpy.types.Operator):
    """Translates n-gons & quads to tris"""
    bl_idname = "object.triangulate"
    bl_label = "Triangulate N-gons & Quads"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        try:
            bpy.ops.object.mode_set(mode='EDIT')
            bpy.ops.mesh.select_all(action='SELECT')
            bpy.ops.mesh.quads_convert_to_tris(quad_method='BEAUTY', ngon_method='BEAUTY')
        except Exception as e:
            self.report({'ERROR'}, f"Error: {e}")
        return {'FINISHED'}


class RepositionModel(bpy.types.Operator):
    """Reposition model (default (0,0,0))(you may need to set your origin to your geometry first)"""
    bl_idname = "object.repositioning_model"
    bl_label = "Zero Model"
    bl_options = {'REGISTER', 'UNDO'}

    count_x: bpy.props.FloatProperty(
        name="X",
        description="Eastings"
    )
    count_y: bpy.props.FloatProperty(
        name="Y",
        description="Northings"
    )
    count_z: bpy.props.FloatProperty(
        name="Z",
        description="AOD/Height"
    )

    def execute(self, context):
        try:
            bpy.context.object.location[0] = self.count_x
            bpy.context.object.location[1] = self.count_y
            bpy.context.object.location[2] = self.count_z
        except Exception as e:
            self.report({'ERROR'}, f"Error: {e}")
        return {'FINISHED'}

class VIEW3D_PT_SelectMesh(bpy.types.Panel):
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "VU.CITY"
    bl_label = "Start Here"

    def draw(self, context):
        self.layout.operator("object.select_mesh", icon = "COLORSET_07_VEC")
        self.layout.operator("object.correct_orientation", icon = "COLORSET_07_VEC")




class VIEW3D_PT_MergeByDistance(bpy.types.Panel):
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "VU.CITY"
    bl_label = "Optimise / Reduce Detail"

    def draw(self, context):
        props_high = self.layout.operator("object.merging_distance", text="High (10cm)", icon="COLORSET_07_VEC")
        props_high.count_cm = 0.1
        props_high = self.layout.operator("object.merging_distance", text="Medium (5cm)", icon="COLORSET_07_VEC")
        props_high.count_cm = 0.05
        props_low = self.layout.operator("object.merging_distance", text="Low (1cm)", icon="COLORSET_07_VEC")
        props_low.count_cm = 0.01


class VIEW3D_PT_OriginToGeometry(bpy.types.Panel):
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "VU.CITY"
    bl_label = "Zero & Geolocate Model"

    def draw(self, context):
        self.layout.operator("object.origining_geometry", icon = "COLORSET_07_VEC")
        self.layout.operator("object.repositioning_model", icon="COLORSET_07_VEC")
        layout = self.layout
        ob = context.object
        ob = context.object
        if ob:
            layout.column().prop(ob, "location", text="Geolocate", icon="COLORSET_07_VEC")
        else:
            layout.label(text="No active object to geolocate")



class VIEW3D_PT_FixSmudges(bpy.types.Panel):
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "VU.CITY"
    bl_label = "Appearance"

    def draw(self, context):
        self.layout.operator("object.fixing_smudges", icon = "COLORSET_07_VEC")
        self.layout.operator("object.removing_materials", icon = "COLORSET_07_VEC")


class VIEW3D_PT_RescaleModel(bpy.types.Panel):
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "VU.CITY"
    bl_label = "Rescale Model"

    def draw(self, context):
        self.layout.operator("object.rescaling_model", text="Scale x0.1", icon = "COLORSET_07_VEC")
        self.layout.operator("object.rescaling_model_x10", text="Scale x10", icon="COLORSET_07_VEC")


class VIEW3D_PT_RevealFaces(bpy.types.Panel):
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "VU.CITY"
    bl_label = "Adjust Faces"

    def draw(self, context):
        self.layout.operator("object.revealing_faces", icon = "COLORSET_07_VEC")
        self.layout.operator("object.triangulate", icon = "COLORSET_07_VEC")



class VIEW3D_PT_RotateBy90(bpy.types.Panel):
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "VU.CITY"
    bl_label = "Rotation"

    def draw(self, context):
        self.layout.operator("object.rotating_90", icon = "COLORSET_07_VEC")




blender_classes = [
    SelectMesh,
    FixSmudges,
    RemoveMaterials,
    CorrectOrientation,
    MergeByDistance,
    OriginToGeometry,
    RescaleModel,
    RescaleModel_x10,
    RevealFaces,
    Triangulate,
    RepositionModel,
    VIEW3D_PT_SelectMesh,
    VIEW3D_PT_FixSmudges,
    VIEW3D_PT_MergeByDistance,
    VIEW3D_PT_RescaleModel,
    VIEW3D_PT_RevealFaces,
    VIEW3D_PT_OriginToGeometry
]

def register():
    for cls in blender_classes:
        bpy.utils.register_class(cls)

def unregister():
    for cls in blender_classes:
        bpy.utils.unregister_class(cls)

if __name__ == "__main__":
    register()